<?php

/**
 * Handles the TeleHealthUserRepository Unit Tests
 *
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2022 Comlink Inc <https://comlinkinc.com/>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace Comlink\OpenEMR\Modules\TeleHealthModule;

use Comlink\OpenEMR\Modules\TeleHealthModule\Models\TeleHealthUser;
use Comlink\OpenEMR\Modules\TeleHealthModule\Repository\TeleHealthUserRepository;
use OpenEMR\Common\Database\QueryUtils;
use PHPUnit\Framework\TestCase;

class TeleHealthUserRepositoryTest extends TestCase
{
    const TEST_USERNAME = "phpunit-test-username";
    const TEST_PASSWORD = "randomToken";

    protected function tearDown(): void
    {
        parent::tearDown(); // TODO: Change the autogenerated stub
        QueryUtils::sqlStatementThrowException("DELETE FROM " . TeleHealthUserRepository::TABLE_NAME
            . " WHERE username LIKE ?", ["%" . self::TEST_USERNAME . "%"]);
    }

    public function testSaveUserWithEmptyUsernameThrowsException()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage("username cannot be empty");

        $user = new TeleHealthUser();
        $user->setAuthToken(self::TEST_PASSWORD);
        $repo = new TeleHealthUserRepository();
        $repo->saveUser($user);
    }

    public function testSaveUserWithEmptyAuthTokenThrowsException()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage("authToken cannot be empty");

        $user = new TeleHealthUser();
        $user->setUsername(self::TEST_USERNAME);
        $repo = new TeleHealthUserRepository();
        $repo->saveUser($user);
    }

    public function testSaveUserCreatesProviderDatabaseRecord()
    {
        $user = new TeleHealthUser();
        $user->setUsername(self::TEST_USERNAME);
        $user->setAuthToken(self::TEST_PASSWORD);
        $user->setDbRecordId(1);
        $user->setIsActive(true);
        $user->setIsPatient(false);
        $user->setDateRegistered(new \DateTime());

        $repo = new TeleHealthUserRepository();
        $repo->saveUser($user);

        $query = "select * from " . TeleHealthUserRepository::TABLE_NAME . " WHERE username = ? ";
        $records = QueryUtils::fetchRecords($query, [self::TEST_USERNAME]);
        $this->assertNotEmpty($records, "Record should have been created");
        $record = $records[0];
        $this->assertEquals($user->getDbRecordId(), $record['user_id']);
        $this->assertEquals(1, $record['active']);
        $this->assertEquals($user->getAuthToken(), $record['auth_token']);
        $this->assertNotEmpty($record['date_created']);
        $this->assertNotEmpty($record['date_updated']);
        $this->assertNotEmpty($record['date_registered']);
    }

    public function testGetUser()
    {
        $user = new TeleHealthUser();
        $user->setUsername(self::TEST_USERNAME);
        $user->setAuthToken(self::TEST_PASSWORD);
        $user->setDbRecordId(1);
        $user->setIsActive(true);
        $user->setIsPatient(false);
        $user->setDateRegistered(new \DateTime());

        $repo = new TeleHealthUserRepository();
        $repo->saveUser($user);

        $repo = new TeleHealthUserRepository();
        $checkUser = $repo->getUser($user->getUsername());

        $this->assertNotNull($checkUser);
        $this->assertNotNull($checkUser->getId());
        $this->assertEquals($user->getUsername(), $checkUser->getUsername());
        $this->assertEquals($user->getDbRecordId(), $checkUser->getDbRecordId());
        $this->assertEquals($user->getAuthToken(), $checkUser->getAuthToken());
        $this->assertNotNull($checkUser->getDateRegistered());
        $this->assertNotNull($checkUser->getDateCreated());
        $this->assertNotNull($checkUser->getDateUpdated());
    }
}
